"use client";

import React, { useState } from "react";
import { useRouter } from "next/navigation";
import Link from "next/link";
import { zodResolver } from "@hookform/resolvers/zod";
import { useForm } from "react-hook-form";
import * as z from "zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Card } from "@/components/ui/card";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { TypographyH3, TypographyP } from "@/components/ui/typography";
import { authService } from "@/services/authService";
import { Loader2, Mail, ArrowLeft, CheckCircle2 } from "lucide-react";

const forgotPasswordSchema = z.object({
  email: z.string().email("الرجاء إدخال عنوان بريد إلكتروني صحيح"),
});

type ForgotPasswordFormValues = z.infer<typeof forgotPasswordSchema>;

const LoadingSpinner = () => (
  <Loader2 className="h-5 w-5 animate-spin" />
);

export default function ForgotPasswordPage() {
  const router = useRouter();
  const [isLoading, setIsLoading] = useState(false);
  const [isSuccess, setIsSuccess] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");

  const form = useForm<ForgotPasswordFormValues>({
    resolver: zodResolver(forgotPasswordSchema),
    defaultValues: {
      email: "",
    },
  });

  const onSubmit = async (data: ForgotPasswordFormValues) => {
    setIsLoading(true);
    setErrorMessage("");

    try {
      const response = await authService.forgotPassword(data);

      if (response.success) {
        setIsSuccess(true);
      }
    } catch (error) {
      const err = error as Error;
      setErrorMessage(err.message || "Failed to send reset link. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  if (isSuccess) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-gray-50 to-gray-100 px-4 py-12">
        <Card className="w-full max-w-md p-8 space-y-6 text-center">
          <div className="flex justify-center">
            <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center">
              <CheckCircle2 className="h-8 w-8 text-green-600" />
            </div>
          </div>

          <div className="space-y-2">
            <TypographyH3 className="text-2xl font-bold text-gray-900">
              تحقق من بريدك الإلكتروني
            </TypographyH3>
            <TypographyP className="text-gray-600">
              لقد أرسلنا رابط إعادة تعيين كلمة المرور إلى عنوان بريدك الإلكتروني. يرجى التحقق من صندوق الوارد الخاص بك واتباع التعليمات لإعادة تعيين كلمة المرور.
            </TypographyP>
          </div>

          <div className="space-y-3 pt-4">
            <Button
              onClick={() => router.push("/login")}
              className="w-full h-12 bg-black text-white rounded-full hover:bg-gray-800 transition-colors duration-200"
            >
              العودة إلى تسجيل الدخول
            </Button>

            <Button
              onClick={() => {
                setIsSuccess(false);
                form.reset();
              }}
              variant="outline"
              className="w-full h-12 rounded-full"
            >
              إرسال رابط آخر
            </Button>
          </div>

          <TypographyP className="text-sm text-gray-500">
            لم تتلق البريد الإلكتروني؟ تحقق من مجلد البريد العشوائي أو حاول مرة أخرى.
          </TypographyP>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-gray-50 to-gray-100 px-4 py-12">
      <Card className="w-full max-w-md p-8 space-y-6">
        {/* Back to Login Link */}
        <Link
          href="/login"
          className="inline-flex items-center text-sm text-gray-600 hover:text-gray-900 transition-colors duration-200"
        >
          <ArrowLeft className="h-4 w-4 ml-2" />
          العودة إلى تسجيل الدخول
        </Link>

        {/* Header */}
        <div className="space-y-2 text-center">
          <div className="flex justify-center mb-4">
            <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
              <Mail className="h-6 w-6 text-blue-600" />
            </div>
          </div>
          <TypographyH3 className="text-2xl font-bold text-gray-900">
            هل نسيت كلمة المرور؟
          </TypographyH3>
          <TypographyP className="text-gray-600">
            لا تقلق! أدخل عنوان بريدك الإلكتروني وسنرسل لك رابطاً لإعادة تعيين كلمة المرور.
          </TypographyP>
        </div>

        {/* Error Message */}
        {errorMessage && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
            <p className="text-sm text-red-600">{errorMessage}</p>
          </div>
        )}

        {/* Form */}
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
            <FormField
              control={form.control}
              name="email"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="text-gray-700 font-medium">
                    البريد الإلكتروني
                  </FormLabel>
                  <FormControl>
                    <Input
                      type="email"
                      placeholder="أدخل بريدك الإلكتروني"
                      className="h-12 rounded-lg border-gray-300 focus:border-black focus:ring-black"
                      {...field}
                      disabled={isLoading}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Submit Button */}
            <Button
              type="submit"
              disabled={isLoading}
              className="w-full h-12 bg-black text-white rounded-full hover:bg-gray-800 transition-colors duration-200 text-[16px] font-medium"
            >
              {isLoading ? (
                <div className="flex items-center justify-center space-x-2">
                  <LoadingSpinner />
                  <span>جاري الإرسال...</span>
                </div>
              ) : (
                "إرسال رابط إعادة التعيين"
              )}
            </Button>
          </form>
        </Form>

        {/* Additional Info */}
        <div className="text-center">
          <TypographyP className="text-sm text-gray-500">
            تتذكر كلمة المرور؟{" "}
            <Link
              href="/login"
              className="text-black font-medium hover:text-gray-700 transition-colors duration-200"
            >
              تسجيل الدخول
            </Link>
          </TypographyP>
        </div>
      </Card>
    </div>
  );
}
