"use client";
import React, { useEffect, useState } from "react";
import { Container } from "@/components/ui/container";
import { TypographyH3 } from "@/components/ui/typography";
import PackageCard from "@/components/PackageCard";
import { packageService } from "@/services/packageService";
import type { PackageDeal } from "@/types/package";
import { Package, Tag, TrendingDown } from "lucide-react";

const PackageDealsPage: React.FC = () => {
  const [packages, setPackages] = useState<PackageDeal[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    const fetchPackages = async () => {
      try {
        setLoading(true);
        const response = await packageService.getPackageDeals({
          available_only: true,
          per_page: 20,
        });
        setPackages(response.data);
      } catch (err) {
        console.error('Error fetching packages:', err);
        setError('Failed to load package deals');
      } finally {
        setLoading(false);
      }
    };

    fetchPackages();
  }, []);

  if (loading) {
    return (
      <section className="py-12 px-4 md:px-6">
        <Container>
          <div className="animate-pulse">
            <div className="h-12 bg-gray-200 rounded w-1/3 mb-8"></div>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
              {Array.from({ length: 6 }).map((_, i) => (
                <div key={i} className="space-y-4">
                  <div className="aspect-square bg-gray-200 rounded"></div>
                  <div className="h-4 bg-gray-200 rounded w-3/4"></div>
                  <div className="h-4 bg-gray-200 rounded w-1/2"></div>
                </div>
              ))}
            </div>
          </div>
        </Container>
      </section>
    );
  }

  if (error) {
    return (
      <section className="py-12 px-4 md:px-6">
        <Container>
          <div className="text-center py-20">
            <Package className="w-16 h-16 mx-auto text-gray-400 mb-4" />
            <h2 className="text-2xl font-bold text-gray-900 mb-2">Error Loading Packages</h2>
            <p className="text-gray-600">{error}</p>
          </div>
        </Container>
      </section>
    );
  }

  if (packages.length === 0) {
    return (
      <section className="py-12 px-4 md:px-6">
        <Container>
          <div className="text-center py-20">
            <Package className="w-16 h-16 mx-auto text-gray-400 mb-4" />
            <h2 className="text-2xl font-bold text-gray-900 mb-2">No Package Deals Available</h2>
            <p className="text-gray-600">Check back soon for exciting bundle offers!</p>
          </div>
        </Container>
      </section>
    );
  }

  return (
    <section className="py-12 px-4 md:px-6">
      <Container>
        {/* Header */}
        <div className="mb-10 text-center">
          <div className="flex items-center justify-center gap-3 mb-4">
            <Tag className="w-10 h-10 text-orange-500" />
            <TypographyH3 className="text-[36px] sm:text-[47px] leading-10 sm:leading-[59px] font-bold prompt tracking-[-0.94px] text-black">
              عروض <span className="text-[#F5411C]">الباقات</span>
            </TypographyH3>
            <Package className="w-10 h-10 text-orange-500" />
          </div>
          <p className="text-gray-600 text-lg max-w-2xl mx-auto">
            وفر أكثر عند شراء المنتجات معاً! اكتشف باقاتنا المميزة بأسعار خاصة
          </p>
          <div className="flex items-center justify-center gap-2 mt-4">
            <TrendingDown className="w-5 h-5 text-green-600" />
            <span className="text-green-600 font-semibold">
              Save up to 30% with package deals
            </span>
          </div>
        </div>

        {/* Stats Bar */}
        <div className="grid grid-cols-3 gap-4 mb-10 max-w-3xl mx-auto">
          <div className="bg-gradient-to-br from-orange-50 to-orange-100 rounded-lg p-4 text-center">
            <div className="text-3xl font-bold text-orange-600">{packages.length}</div>
            <div className="text-sm text-gray-600">Available Packages</div>
          </div>
          <div className="bg-gradient-to-br from-green-50 to-green-100 rounded-lg p-4 text-center">
            <div className="text-3xl font-bold text-green-600">
              {Math.max(...packages.map(p => p.discount_percentage))}%
            </div>
            <div className="text-sm text-gray-600">Max Discount</div>
          </div>
          <div className="bg-gradient-to-br from-blue-50 to-blue-100 rounded-lg p-4 text-center">
            <div className="text-3xl font-bold text-blue-600">
              {packages.reduce((sum, p) => sum + (p.products?.length || 0), 0)}
            </div>
            <div className="text-sm text-gray-600">Total Products</div>
          </div>
        </div>

        {/* Package Grid */}
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6 lg:gap-8">
          {packages.map((packageDeal) => (
            <PackageCard 
              key={packageDeal.id} 
              packageDeal={packageDeal}
            />
          ))}
        </div>

        {/* Info Section */}
        <div className="mt-16 bg-gradient-to-r from-orange-50 to-red-50 rounded-2xl p-8 text-center">
          <h3 className="text-2xl font-bold text-gray-900 mb-4">
            لماذا عروض الباقات؟
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mt-8">
            <div className="space-y-2">
              <div className="w-12 h-12 bg-orange-500 rounded-full flex items-center justify-center mx-auto">
                <TrendingDown className="w-6 h-6 text-white" />
              </div>
              <h4 className="font-semibold text-lg">Save More</h4>
              <p className="text-gray-600 text-sm">Get better prices when buying products together</p>
            </div>
            <div className="space-y-2">
              <div className="w-12 h-12 bg-orange-500 rounded-full flex items-center justify-center mx-auto">
                <Package className="w-6 h-6 text-white" />
              </div>
              <h4 className="font-semibold text-lg">Complete Sets</h4>
              <p className="text-gray-600 text-sm">Curated product combinations that work great together</p>
            </div>
            <div className="space-y-2">
              <div className="w-12 h-12 bg-orange-500 rounded-full flex items-center justify-center mx-auto">
                <Tag className="w-6 h-6 text-white" />
              </div>
              <h4 className="font-semibold text-lg">Limited Time</h4>
              <p className="text-gray-600 text-sm">Special offers available for a limited period</p>
            </div>
          </div>
        </div>
      </Container>
    </section>
  );
};

export default PackageDealsPage;
