"use client";

import React, { useState, useEffect, Suspense } from "react";
import { useRouter, useSearchParams } from "next/navigation";

// Disable static generation for this page
export const dynamic = 'force-dynamic';
import Link from "next/link";
import { zodResolver } from "@hookform/resolvers/zod";
import { useForm } from "react-hook-form";
import * as z from "zod";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Card } from "@/components/ui/card";
import { TypographyH3, TypographyP } from "@/components/ui/typography";
import { authService } from "@/services/authService";
import { Loader2, Lock, CheckCircle2, Eye, EyeOff } from "lucide-react";

const resetPasswordSchema = z.object({
  email: z.string().email("الرجاء إدخال عنوان بريد إلكتروني صحيح"),
  password: z
    .string()
    .min(8, "يجب أن تتكون كلمة المرور من 8 أحرف على الأقل")
    .regex(/[A-Z]/, "يجب أن تحتوي كلمة المرور على حرف كبير واحد على الأقل")
    .regex(/[a-z]/, "يجب أن تحتوي كلمة المرور على حرف صغير واحد على الأقل")
    .regex(/[0-9]/, "يجب أن تحتوي كلمة المرور على رقم واحد على الأقل"),
  password_confirmation: z.string(),
}).refine((data) => data.password === data.password_confirmation, {
  message: "كلمات المرور غير متطابقة",
  path: ["password_confirmation"],
});

type ResetPasswordFormValues = z.infer<typeof resetPasswordSchema>;

const LoadingSpinner = () => (
  <Loader2 className="h-5 w-5 animate-spin" />
);

function ResetPasswordContent() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const [isLoading, setIsLoading] = useState(false);
  const [isSuccess, setIsSuccess] = useState(false);
  const [errorMessage, setErrorMessage] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [token, setToken] = useState("");
  const [email, setEmail] = useState("");

  useEffect(() => {
    const tokenParam = searchParams.get("token");
    const emailParam = searchParams.get("email");

    if (!tokenParam || !emailParam) {
      setErrorMessage("رابط إعادة التعيين غير صالح. يرجى طلب إعادة تعيين كلمة مرور جديدة.");
      return;
    }

    setToken(tokenParam);
    setEmail(decodeURIComponent(emailParam));
  }, [searchParams]);

  const form = useForm<ResetPasswordFormValues>({
    resolver: zodResolver(resetPasswordSchema),
    defaultValues: {
      email: "",
      password: "",
      password_confirmation: "",
    },
  });

  // Update email field when email state changes
  useEffect(() => {
    if (email) {
      form.setValue("email", email);
    }
  }, [email, form]);

  const onSubmit = async (data: ResetPasswordFormValues) => {
    if (!token) {
      setErrorMessage("رمز إعادة التعيين غير صالح. يرجى طلب إعادة تعيين كلمة مرور جديدة.");
      return;
    }

    setIsLoading(true);
    setErrorMessage("");

    try {
      const response = await authService.resetPassword({
        token,
        email: data.email,
        password: data.password,
        password_confirmation: data.password_confirmation,
      });

      if (response.success) {
        setIsSuccess(true);
        setTimeout(() => {
          router.push("/login");
        }, 3000);
      }
    } catch (error) {
      const err = error as Error;
      setErrorMessage(err.message || "Failed to reset password. Please try again.");
    } finally {
      setIsLoading(false);
    }
  };

  if (isSuccess) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-gray-50 to-gray-100 px-4 py-12">
        <Card className="w-full max-w-md p-8 space-y-6 text-center">
          <div className="flex justify-center">
            <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center">
              <CheckCircle2 className="h-8 w-8 text-green-600" />
            </div>
          </div>

          <div className="space-y-2">
            <TypographyH3 className="text-2xl font-bold text-gray-900">
              تم إعادة تعيين كلمة المرور بنجاح!
            </TypographyH3>
            <TypographyP className="text-gray-600">
              تم إعادة تعيين كلمة المرور بنجاح. يمكنك الآن تسجيل الدخول باستخدام كلمة المرور الجديدة.
            </TypographyP>
          </div>

          <div className="pt-4">
            <Button
              onClick={() => router.push("/login")}
              className="w-full h-12 bg-black text-white rounded-full hover:bg-gray-800 transition-colors duration-200"
            >
              المتابعة إلى تسجيل الدخول
            </Button>
          </div>

          <TypographyP className="text-sm text-gray-500">
            جاري التوجيه إلى صفحة تسجيل الدخول خلال 3 ثوان...
          </TypographyP>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-gray-50 to-gray-100 px-4 py-12">
      <Card className="w-full max-w-md p-8 space-y-6">
        {/* Header */}
        <div className="space-y-2 text-center">
          <div className="flex justify-center mb-4">
            <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
              <Lock className="h-6 w-6 text-blue-600" />
            </div>
          </div>
          <TypographyH3 className="text-2xl font-bold text-gray-900">
            إعادة تعيين كلمة المرور
          </TypographyH3>
          <TypographyP className="text-gray-600">
            أدخل كلمة المرور الجديدة أدناه
          </TypographyP>
        </div>

        {/* Error Message */}
        {errorMessage && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
            <p className="text-sm text-red-600">{errorMessage}</p>
            {errorMessage.includes("غير صالح") && (
              <Link
                href="/forgot-password"
                className="text-sm text-red-700 font-medium hover:underline mt-2 inline-block"
              >
                طلب رابط إعادة تعيين جديد
              </Link>
            )}
          </div>
        )}

        {/* Form */}
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-5">
            <FormField
              control={form.control}
              name="email"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="text-gray-700 font-medium">
                    البريد الإلكتروني
                  </FormLabel>
                  <FormControl>
                    <Input
                      type="email"
                      placeholder="بريدك الإلكتروني"
                      className="h-12 rounded-lg border-gray-300 focus:border-black focus:ring-black bg-gray-50"
                      {...field}
                      disabled={true}
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="password"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="text-gray-700 font-medium">
                    كلمة المرور الجديدة
                  </FormLabel>
                  <FormControl>
                    <div className="relative">
                      <Input
                        type={showPassword ? "text" : "password"}
                        placeholder="أدخل كلمة المرور الجديدة"
                        className="h-12 rounded-lg border-gray-300 focus:border-black focus:ring-black pr-10"
                        {...field}
                        disabled={isLoading}
                      />
                      <button
                        type="button"
                        onClick={() => setShowPassword(!showPassword)}
                        className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700"
                      >
                        {showPassword ? (
                          <EyeOff className="h-5 w-5" />
                        ) : (
                          <Eye className="h-5 w-5" />
                        )}
                      </button>
                    </div>
                  </FormControl>
                  <FormMessage />
                  <p className="text-xs text-gray-500 mt-1">
                    يجب أن تتكون من 8 أحرف على الأقل وتحتوي على أحرف كبيرة وصغيرة وأرقام
                  </p>
                </FormItem>
              )}
            />

            <FormField
              control={form.control}
              name="password_confirmation"
              render={({ field }) => (
                <FormItem>
                  <FormLabel className="text-gray-700 font-medium">
                    تأكيد كلمة المرور
                  </FormLabel>
                  <FormControl>
                    <div className="relative">
                      <Input
                        type={showConfirmPassword ? "text" : "password"}
                        placeholder="أعد إدخال كلمة المرور"
                        className="h-12 rounded-lg border-gray-300 focus:border-black focus:ring-black pr-10"
                        {...field}
                        disabled={isLoading}
                      />
                      <button
                        type="button"
                        onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                        className="absolute left-3 top-1/2 -translate-y-1/2 text-gray-500 hover:text-gray-700"
                      >
                        {showConfirmPassword ? (
                          <EyeOff className="h-5 w-5" />
                        ) : (
                          <Eye className="h-5 w-5" />
                        )}
                      </button>
                    </div>
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            {/* Submit Button */}
            <Button
              type="submit"
              disabled={isLoading}
              className="w-full h-12 bg-black text-white rounded-full hover:bg-gray-800 transition-colors duration-200 text-[16px] font-medium"
            >
              {isLoading ? (
                <div className="flex items-center justify-center space-x-2">
                  <LoadingSpinner />
                  <span>جاري إعادة التعيين...</span>
                </div>
              ) : (
                "إعادة تعيين كلمة المرور"
              )}
            </Button>
          </form>
        </Form>

        {/* Additional Info */}
        <div className="text-center">
          <TypographyP className="text-sm text-gray-500">
            تذكرت كلمة المرور؟{" "}
            <Link
              href="/login"
              className="text-black font-medium hover:text-gray-700 transition-colors duration-200"
            >
              تسجيل الدخول
            </Link>
          </TypographyP>
        </div>
      </Card>
    </div>
  );
}

export default function ResetPasswordPage() {
  return (
    <Suspense
      fallback={
        <div className="min-h-screen flex items-center justify-center">
          <Loader2 className="h-8 w-8 animate-spin text-primary" />
        </div>
      }
    >
      <ResetPasswordContent />
    </Suspense>
  );
}
