/**
 * Cache Status Component
 * Shows cache information and allows manual cache control
 * This is useful for debugging and ensuring cache invalidation works
 */

"use client";

import React, { useState, useEffect } from 'react';
import { cache } from '@/lib/cache';
import { invalidateCache } from '@/lib/cache-invalidation';

export default function CacheStatus() {
  const [cacheStats, setCacheStats] = useState({ totalItems: 0, totalSize: 0 });
  const [lastUpdate, setLastUpdate] = useState<string>('');
  const [isVisible, setIsVisible] = useState(false);

  const updateStats = () => {
    const stats = cache.getStats();
    setCacheStats(stats);
    setLastUpdate(new Date().toLocaleTimeString());
  };

  useEffect(() => {
    updateStats();
    
    // Update stats every 5 seconds
    const interval = setInterval(updateStats, 5000);
    
    // Listen for cache invalidation events
    const handleCacheUpdate = () => {
      setTimeout(updateStats, 100); // Small delay to ensure cache is updated
    };
    
    window.addEventListener('productUpdated', handleCacheUpdate);
    window.addEventListener('categoryUpdated', handleCacheUpdate);
    
    return () => {
      clearInterval(interval);
      window.removeEventListener('productUpdated', handleCacheUpdate);
      window.removeEventListener('categoryUpdated', handleCacheUpdate);
    };
  }, []);

  if (!isVisible) {
    return (
      <button
        onClick={() => setIsVisible(true)}
        className="fixed bottom-4 right-4 bg-gray-800 text-white p-2 rounded-full text-xs z-50 hover:bg-gray-700"
        title="Show cache status"
      >
        📊
      </button>
    );
  }

  return (
    <div className="fixed bottom-4 right-4 bg-white border border-gray-300 rounded-lg p-4 shadow-lg z-50 text-sm max-w-xs">
      <div className="flex justify-between items-center mb-2">
        <h3 className="font-bold text-gray-800">Cache Status</h3>
        <button
          onClick={() => setIsVisible(false)}
          className="text-gray-500 hover:text-gray-700"
        >
          ✕
        </button>
      </div>
      
      <div className="space-y-2 text-xs text-gray-600">
        <div>Items: {cacheStats.totalItems}</div>
        <div>Size: {Math.round(cacheStats.totalSize / 1024)}KB</div>
        <div>Updated: {lastUpdate}</div>
      </div>
      
      <div className="mt-3 space-y-1">
        <button
          onClick={() => {
            invalidateCache.all();
            updateStats();
          }}
          className="w-full bg-red-500 text-white px-2 py-1 rounded text-xs hover:bg-red-600"
        >
          Clear All Cache
        </button>
        
        <button
          onClick={() => {
            invalidateCache.product('test', 'test-slug', 1);
            updateStats();
          }}
          className="w-full bg-blue-500 text-white px-2 py-1 rounded text-xs hover:bg-blue-600"
        >
          Test Product Update
        </button>
        
        <button
          onClick={updateStats}
          className="w-full bg-gray-500 text-white px-2 py-1 rounded text-xs hover:bg-gray-600"
        >
          Refresh Stats
        </button>
      </div>
    </div>
  );
}