"use client";

import React from "react";
import Link from "next/link";
import { useCategories } from "@/hooks/useCategories";
import { Container } from "@/components/ui/container";
import { TypographyH3 } from "@/components/ui/typography";

interface CategoriesNavigationProps {
  showTitle?: boolean;
  className?: string;
}

const CategoriesNavigation: React.FC<CategoriesNavigationProps> = ({ 
  showTitle = true, 
  className = "" 
}) => {
  const { categories, loading, error } = useCategories();

  if (loading) {
    return (
      <section className={`py-8 ${className}`}>
        <Container>
          {showTitle && (
            <TypographyH3 className="text-2xl font-bold mb-6">
              Shop by Categories
            </TypographyH3>
          )}
          <div className="flex items-center justify-center min-h-[100px]">
            <p className="text-gray-500">Loading categories...</p>
          </div>
        </Container>
      </section>
    );
  }

  if (error) {
    return (
      <section className={`py-8 ${className}`}>
        <Container>
          {showTitle && (
            <TypographyH3 className="text-2xl font-bold mb-6">
              Shop by Categories
            </TypographyH3>
          )}
          <p className="text-red-500">Error loading categories: {error}</p>
        </Container>
      </section>
    );
  }

  // Get parent categories with their children
  const parentCategories = categories.filter(cat => cat.parent_id === null);

  return (
    <section className={`py-8 ${className}`}>
      <Container>
        {/* {showTitle && (
          <TypographyH3 className="text-[36px] sm:text-[47px] leading-10 sm:leading-[59px] font-bold prompt tracking-[-0.94px] text-black mb-8">
            Shop by <span className="text-[#F5411C]">Categories</span>
          </TypographyH3>
        )} */}
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
          {parentCategories.map((category) => (
            <div key={category.id} className="group">
              <div className="bg-gray-50 hover:bg-gray-100 transition-colors p-6 rounded-lg border">
                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                  {category.name}
                </h3>
                
                {category.children && category.children.length > 0 && (
                  <ul className="space-y-2">
                    {category.children.slice(0, 5).map((child) => (
                      <li key={child.id}>
                        <Link 
                          href={`/products?category=${child.id}`}
                          className="text-sm text-gray-600 hover:text-[#F5411C] transition-colors block py-1"
                        >
                          {child.name}
                        </Link>
                      </li>
                    ))}
                    {category.children.length > 5 && (
                      <li>
                        <Link 
                          href={`/categories/${category.slug}`}
                          className="text-sm text-[#F5411C] hover:text-[#e63914] font-medium"
                        >
                          View all ({category.children.length})
                        </Link>
                      </li>
                    )}
                  </ul>
                )}
                
                {(!category.children || category.children.length === 0) && (
                  <Link 
                    href={`/products?category=${category.id}`}
                    className="text-sm text-[#F5411C] hover:text-[#e63914] font-medium"
                  >
                    Shop Now →
                  </Link>
                )}
              </div>
            </div>
          ))}
        </div>
      </Container>
    </section>
  );
};

export default CategoriesNavigation;