"use client";
import React, { useState, useEffect } from "react";
import { Container } from "@/components/ui/container";
import { TypographyH3 } from "@/components/ui/typography";
import { Button } from "@/components/ui/button";
import PackageCard from "@/components/PackageCard";
import { ChevronLeft, ChevronRight, Package } from "lucide-react";
import { packageService } from "@/services/packageService";
import type { PackageDeal } from "@/types/package";
import Link from "next/link";

const FeaturedPackages: React.FC = () => {
  const [packages, setPackages] = useState<PackageDeal[]>([]);
  const [loading, setLoading] = useState(true);
  const [currentIndex, setCurrentIndex] = useState(0);

  useEffect(() => {
    const fetchPackages = async () => {
      try {
        const data = await packageService.getFeaturedPackages();
        setPackages(data);
      } catch (error) {
        console.error('Error fetching featured packages:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchPackages();
  }, []);

  const scrollPrev = () => {
    setCurrentIndex((prev) => Math.max(0, prev - 1));
  };

  const scrollNext = () => {
    setCurrentIndex((prev) => Math.min(packages.length - 3, prev + 1));
  };

  if (loading) {
    return (
      <section className="py-12 px-4 md:px-6 bg-gradient-to-b from-orange-50/50 to-white">
        <Container>
          <div className="animate-pulse">
            <div className="h-12 bg-gray-200 rounded w-1/3 mb-8"></div>
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-6">
              {Array.from({ length: 3 }).map((_, i) => (
                <div key={i} className="h-96 bg-gray-200 rounded-lg"></div>
              ))}
            </div>
          </div>
        </Container>
      </section>
    );
  }

  if (packages.length === 0) {
    return null;
  }

  return (
    <section className="py-12 px-4 md:px-6 bg-gradient-to-b from-orange-50/50 to-white">
      <Container>
        <div className="flex items-center justify-between gap-4 mb-6 lg:mb-[46px]">
          <div className="flex items-center gap-3">
            <Package className="w-8 h-8 text-[#F5411C]" />
            <TypographyH3 className="text-[36px] sm:text-[47px] leading-10 sm:leading-[59px] font-bold prompt tracking-[-0.94px] text-black">
              عروض <span className="text-[#F5411C]">الباقات</span>
            </TypographyH3>
          </div>
          <div className="flex items-center gap-4">
            <Link href="/packages">
              <Button
                variant="outline"
                className="hidden sm:inline-flex border-[#F5411C] text-[#F5411C] hover:bg-[#F5411C] hover:text-white"
              >
                View All Packages
              </Button>
            </Link>
            <div className="flex gap-2">
              <Button
                variant="outline"
                size="icon"
                className="w-10 sm:w-[55px] h-10 sm:h-[55px] rounded-full border-black hover:bg-black hover:text-white transition-colors"
                onClick={scrollPrev}
                disabled={currentIndex === 0}
              >
                <ChevronLeft className="size-6" />
              </Button>
              <Button
                variant="outline"
                size="icon"
                className="w-10 sm:w-[55px] h-10 sm:h-[55px] rounded-full border-black hover:bg-black hover:text-white transition-colors"
                onClick={scrollNext}
                disabled={currentIndex >= packages.length - 3}
              >
                <ChevronRight className="size-6" />
              </Button>
            </div>
          </div>
        </div>

        {/* Desktop: Sliding Grid */}
        <div className="hidden lg:block overflow-hidden">
          <div
            className="flex transition-transform duration-500 ease-in-out gap-6"
            style={{
              transform: `translateX(-${currentIndex * (100 / 3)}%)`,
            }}
          >
            {packages.map((packageDeal) => (
              <div key={packageDeal.id} className="flex-shrink-0 w-1/3 px-3">
                <PackageCard packageDeal={packageDeal} />
              </div>
            ))}
          </div>
        </div>

        {/* Mobile/Tablet: Grid */}
        <div className="grid lg:hidden grid-cols-1 sm:grid-cols-2 gap-6">
          {packages.slice(0, 4).map((packageDeal) => (
            <PackageCard key={packageDeal.id} packageDeal={packageDeal} />
          ))}
        </div>

        {/* Mobile View All Button */}
        <div className="mt-8 text-center sm:hidden">
          <Link href="/packages">
            <Button
              variant="outline"
              className="w-full border-[#F5411C] text-[#F5411C] hover:bg-[#F5411C] hover:text-white"
            >
              <Package className="w-4 h-4 mr-2" />
              View All Package Deals
            </Button>
          </Link>
        </div>
      </Container>
    </section>
  );
};

export default FeaturedPackages;
