"use client";
import React from "react";

import { Container } from "@/components/ui/container";
import ProductCard from "./ProductCard";
import type { Product } from "@/types/product";
import { TypographyH3 } from "./ui/typography";
import { useProducts } from "@/hooks/useProducts";
import { convertApiProductToProduct } from "@/types/api";

const FlashSaleForBestSellers: React.FC = () => {
    // Fetch products for flash sale (limit to first 6)
    const { products: apiProducts, loading } = useProducts();
    
    // Convert API products to frontend format and add sale information
    const flashSaleProducts = React.useMemo(() => {
        return apiProducts.slice(0, 6).map((apiProduct, index) => {
            const product = convertApiProductToProduct(apiProduct);
            
            // Add sale pricing logic
            const originalPrice = parseFloat(product.price.replace('$', ''));
            const salePercentages = [8, 13, 15, 20, 25, 30];
            const salePercentage = salePercentages[index % salePercentages.length];
            const salePrice = originalPrice * (1 - salePercentage / 100);
            
            return {
                ...product,
                price: `$${salePrice.toFixed(2)}`,
                oldPrice: product.price,
                saleLabel: `${salePercentage}% Off`,
            };
        });
    }, [apiProducts]);

    if (loading) {
        return (
            <section className="px-4 md:px-6 py-8 sm:py-12">
                <Container>
                    <TypographyH3 className="text-[36px] sm:text-[47px] leading-10 sm:leading-[59px] font-bold prompt tracking-[-0.94px] text-black">
                        تخفيضات خاطفة ! <span className="text-[#F5411C]"> على</span> الأكثر مبيعاً
                    </TypographyH3>
                    <div className="mt-8 flex items-center justify-center min-h-[300px]">
                        <p className="text-gray-500">Loading flash sale products...</p>
                    </div>
                </Container>
            </section>
        );
    }

    return (
        <section className="px-4 md:px-6 py-8 sm:py-12">
            <Container>
                <TypographyH3 className="text-[36px] sm:text-[47px] leading-10 sm:leading-[59px] font-bold prompt tracking-[-0.94px] text-black">
                        تخفيضات خاطفة ! <span className="text-[#F5411C]"> على</span> الأكثر مبيعاً
                </TypographyH3>

                <div className="mt-8 sm:mt-10 lg:mt-[65px] grid grid-cols-2 sm:grid-cols-2 lg:grid-cols-3 gap-5">
                    {flashSaleProducts.map((p: Product) => (
                        <ProductCard
                            key={p.id}
                            id={p.id}
                            slug={p.slug}
                            title={p.title}
                            price={p.price}
                            oldPrice={p.oldPrice}
                            imageSrc={p.imageSrc}
                            saleLabel={p.saleLabel}
                            rating={p.rating || 4.5}
                            ratingCount={p.ratingCount || 100}
                            variant="flash"
                        />
                    ))}
                </div>
            </Container>
        </section>
    );
};

export default FlashSaleForBestSellers;