"use client";

import React from "react";
import { Button } from "@/components/ui/button";
import { OrderItem } from "@/types/order";
import Image from "next/image";
import {
  TypographyH3,
  TypographyP,
  TypographySmall,
} from "@/components/ui/typography";
import { Separator } from "@/components/ui/separator";

export const currency = (n: number | string) => `$${Number(n).toFixed(2)}`;

type Props = {
  item: OrderItem;
  canCancel?: boolean;
  variant?: "default" | "cancel";
};

const StarIcons = ({ rating }: { rating: number }) => {
  const stars = Array.from({ length: 5 }, (_, idx) => {
    const i = idx + 1;
    let src = "/assets/icons/star.svg";
    if (rating >= i) src = "/assets/icons/full-star.svg";
    else if (rating >= i - 0.5) src = "/assets/icons/half-star.svg";
    return (
      <Image
        key={idx}
        src={src}
        alt="star"
        width={24.71}
        height={2471}
        className="sm:h-6 sm:w-6 w-3.5 h-[15px]"
      />
    );
  });
  return <div className="flex items-center gap-1 sm:gap-1.5">{stars}</div>;
};

export default function OrderProductCard({
  item,
  canCancel,
  variant = "default",
}: Props) {
  return (
    <div className="flex items-start sm:items-center justify-between">
      <div className="flex items-center gap-[13px] sm:gap-6">
        <div className="h-20 sm:h-28 min-w-20 sm:w-28 overflow-hidden rounded-[5px] sm:rounded-[17px] bg-[#F9FAFB] border border-[#D0D5DD] flex items-center justify-center">
          <Image
            src={item.image}
            alt={item.title}
            width={112}
            height={112}
            className="object-cover w-20 h-20 sm:h-28 sm:w-28"
          />
        </div>
        <div className="flex flex-col gap-[3px] sm:gap-3.5">
          <TypographyH3 className="text-[14px] sm:text-[24px] leading-8 font-bold">
            {item.title}
          </TypographyH3>
          <div className="flex items-center gap-[11px] sm:gap-4 text-sm leading-none">
            <StarIcons rating={item.rating} />
            <TypographySmall className="text-[10px] leading-3 text-black">
              {item.rating}/<span className="text-[#00000099]">5</span>
            </TypographySmall>
          </div>
          <div className="flex items-center flex-wrap gap-y-1">
            {item.attributes.map((a, i) => (
              <React.Fragment key={i}>
                {i !== 0 && (
                  <Separator
                    orientation="vertical"
                    className="mx-1 sm:mx-3 h-4"
                  />
                )}
                <TypographySmall className="text-[#667085] text-[10px] sm:text-[16px] leading-6 font-normal inter">
                  {a}
                </TypographySmall>
              </React.Fragment>
            ))}
          </div>
        </div>
      </div>

      {variant === "cancel" ? (
        <Button
          variant="link"
          className="text-[#BF0000] inter text-[13px] sm:text-[20px] leading-8 sm:leading-[30px] font-semibold h-auto hover:no-underline p-0 sm:p-4"
        >
          Cancel Order
        </Button>
      ) : (
        <div className="flex items-center gap-6">
          <div className="flex flex-col items-end">
            <TypographyP className="text-[#1D2939] text-[14px] sm:text-[20px] leading-8 sm:leading-[30px] font-semibold inter sm:mb-0.5">
              {currency(item.price)}
            </TypographyP>
            <TypographySmall className="text-[#667085] text-[12px] sm:text-[16px] leading-6 font-normal inter">
              Qty: {item.qty}
            </TypographySmall>
          </div>
          {canCancel ? (
            <Button variant="destructive" size="sm">
              Cancel Order
            </Button>
          ) : null}
        </div>
      )}
    </div>
  );
}
