"use client";
import React, { useMemo, useState } from "react";
import Image from "next/image";
import { Container } from "../ui/container";
import {
    Breadcrumb,
    BreadcrumbItem,
    BreadcrumbLink,
    BreadcrumbList,
    BreadcrumbPage,
    BreadcrumbSeparator,
} from "../ui/breadcrumb";
import { TypographyH3, TypographyH4 } from "../ui/typography";
import { Separator } from "../ui/separator";
import { Badge } from "../ui/badge";
import { Button } from "../ui/button";

type PDImage = { id: string; src: string; alt: string };

type ProductDetailsData = {
    id: string;
    title: string;
    price: string;
    oldPrice?: string;
    saleLabel?: string;
    rating: number;
    ratingText?: string;
    description: string;
    images: PDImage[];
    colors: { id: string; name: string; hex: string }[];
    sizes: string[];
};

const sampleProduct: ProductDetailsData = {
    id: "p1",
    title: "One Life Graphic T-shirt",
    price: "$260",
    oldPrice: "$300",
    saleLabel: "-40%",
    rating: 4.5,
    ratingText: "4.5/5",
    description:
        "This graphic t-shirt which is perfect for any occasion. Crafted from a soft and breathable fabric.",
    images: [
        { id: "i1", src: "/assets/shirt-2.png", alt: "T-shirt front" },
        { id: "i2", src: "/assets/shirt.png", alt: "T-shirt back" },
        { id: "i3", src: "/assets/shirt-3.webp", alt: "Model wearing t-shirt" },
    ],
    colors: [
        { id: "c1", name: "Olive", hex: "#4F4631" },
        { id: "c2", name: "Charcoal", hex: "#314F4A" },
        { id: "c3", name: "Navy", hex: "#31344F" },
    ],
    sizes: ["Small", "Medium", "Large", "X-Large"],
};

const renderStars = (rating: number) => {
    const full = Math.floor(rating);
    const half = rating - full >= 0.5;
    return (
        <div className="flex items-center gap-1.5">
            {Array.from({ length: 5 }).map((_, i) => {
                const isFull = i < full;
                const isHalf = i === full && half;
                const src = isFull
                    ? "/assets/icons/full-star.svg"
                    : isHalf
                        ? "/assets/icons/half-star.svg"
                        : "/assets/icons/full-star.svg";
                const cls =
                    isFull || isHalf
                        ? "w-[18.49px] h-[18.49px]"
                        : "w-[18.49px] h-[18.49px] opacity-30";
                const alt = isFull ? "full-star" : isHalf ? "half-star" : "empty-star";
                return (
                    <Image
                        key={i}
                        src={src}
                        alt={alt}
                        width={19}
                        height={19}
                        className={cls}
                    />
                );
            })}
        </div>
    );
};

const ProductDetails: React.FC = () => {
    const p = sampleProduct;
    const [activeImage, setActiveImage] = useState<PDImage>(p.images[0]);
    const [color, setColor] = useState(p.colors[0].id);
    const [size, setSize] = useState<string>(p.sizes[2]);
    const [qty, setQty] = useState<number>(1);

    const disabledMinus = useMemo(() => qty <= 1, [qty]);

    return (
        <div className="w-full px-4 md:px-6 mb-14 md:mb-[60px] overflow-x-hidden">
            <Container className="relative">
                <Image src='/assets/icons/star.svg' alt="star" width={22} height={22} className="absolute -right-5 sm:right-1 xl:-right-10 top-[620px] lg:top-[77px]" />
                <Image src='/assets/icons/line-2.svg' alt="line" width={97} height={6} className="absolute -right-14 sm:right-6 xl:-right-4 top-[680px] lg:top-[130px]" />
                <Breadcrumb className="mb-5 md:mb-9">
                    <BreadcrumbList>
                        <BreadcrumbItem>
                            <BreadcrumbLink href="#">Home</BreadcrumbLink>
                        </BreadcrumbItem>
                        <BreadcrumbSeparator />
                        <BreadcrumbItem>
                            <BreadcrumbLink href="#">Shop</BreadcrumbLink>
                        </BreadcrumbItem>
                        <BreadcrumbSeparator />
                        <BreadcrumbItem>
                            <BreadcrumbLink href="#">Men</BreadcrumbLink>
                        </BreadcrumbItem>
                        <BreadcrumbSeparator />
                        <BreadcrumbItem>
                            <BreadcrumbPage>T‑shirts</BreadcrumbPage>
                        </BreadcrumbItem>
                    </BreadcrumbList>
                </Breadcrumb>

                <div className="grid grid-cols-1 lg:grid-cols-[minmax(0,550px)_minmax(0,1fr)] xl:grid-cols-[minmax(0,610px)_minmax(0,1fr)] gap-6 xl:gap-10">
                    {/* Left: Gallery */}
                    <div className="lg:max-w-[610px] w-full flex sm:flex-row flex-col-reverse items-start gap-3.5">
                        <div className="flex flex-row sm:flex-col gap-3.5 min-w-[152px]">
                            {p.images.map((img) => (
                                <button
                                    key={img.id}
                                    onClick={() => setActiveImage(img)}
                                    className={`rounded-xl sm:rounded-[20px] bg-[#F0EEED] cursor-pointer border ${activeImage.id === img.id
                                        ? "border-black"
                                        : "border-transparent"
                                        }`}
                                >
                                    <div className="w-full h-[100px] sm:h-[165px] rounded-[20px] overflow-hidden flex items-center justify-center bg-[#F0EEED]">
                                        <Image
                                            src={img.src}
                                            alt={img.alt}
                                            width={152}
                                            height={167}
                                            className="object-cover"
                                        />
                                    </div>
                                </button>
                            ))}
                        </div>
                        <div className="rounded-xl sm:rounded-[20px] bg-[#F0EEED] w-full lg:max-w-[444px] flex items-center justify-center overflow-hidden">
                            <Image
                                src={activeImage.src}
                                alt={activeImage.alt}
                                width={444}
                                height={530}
                                className="object-cover h-[400px] sm:h-[530px]"
                            />
                        </div>
                    </div>

                    {/* Right: Details */}
                    <div className="w-full lg:max-w-[590px]">
                        <TypographyH3 className="text-[28px] sm:text-[32px] xl:text-[40px] leading-10 sm:leading-[42px] xl:leading-[60px] font-extrabold prompt">
                            {p.title}
                        </TypographyH3>

                        <div className="mt-2 flex items-center gap-3 sm:gap-4">
                            {renderStars(p.rating)}
                            <span className="text-[#667085] text-sm">{p.ratingText}</span>
                        </div>

                        <div className="mt-3.5 flex items-center gap-3">
                            <span className="text-[28px] md:text-[32px] leading-[39px] font-bold">
                                {p.price}
                            </span>
                            {p.oldPrice ? (
                                <span className="text-[#0000004D] line-through text-[28px] md:text-[32px] leading-[39px] font-bold">
                                    {p.oldPrice}
                                </span>
                            ) : null}
                            {p.saleLabel ? (
                                <Badge className="bg-[#FF33331A] text-[#FF3333] border-0 hover:bg-[#FF33331A] rounded-full text-[16px] leading-[22px] font-medium px-3.5 py-1.5">
                                    {p.saleLabel}
                                </Badge>
                            ) : null}
                        </div>

                        <p className="mt-4 sm:mt-6 text-[#00000099] text-[16px] leading-5 font-normal max-w-[600px]">
                            {p.description}
                        </p>

                        <Separator className="my-4 sm:my-6" />

                        {/* Colors */}
                        <div>
                            <TypographyH4 className="text-[#00000099] text-[16px] leading-[11px] font-normal mb-3 sm:mb-4">
                                Select Colors
                            </TypographyH4>
                            <div className="flex items-center gap-4">
                                {p.colors.map((c) => (
                                    <button
                                        key={c.id}
                                        onClick={() => setColor(c.id)}
                                        aria-label={c.name}
                                        className={`relative cursor-pointer flex items-center justify-center h-[37px] w-[37px] rounded-full transition ${color === c.id ? "" : ""}`}
                                        style={{ backgroundColor: c.hex }}
                                    >
                                        {color === c.id ? (
                                            <Image src='/assets/icons/checkmark.svg' alt="checkmark" width={16} height={16} />
                                        ) : null}
                                    </button>
                                ))}
                            </div>
                        </div>

                        <Separator className="my-4 sm:my-6" />

                        {/* Sizes */}
                        <div>
                            <TypographyH4 className="text-[#00000099] text-[16px] leading-[11px] font-normal mb-3 sm:mb-4">
                                Choose Size
                            </TypographyH4>
                            <div className="flex flex-wrap items-center gap-3">
                                {p.sizes.map((s) => {
                                    const selected = size === s;
                                    return (
                                        <Button
                                            key={s}
                                            type="button"
                                            onClick={() => setSize(s)}
                                            variant={selected ? "default" : "outline"}
                                            className={`${selected ? "bg-black text-white font-medium" : "bg-[#F0F0F0] text-[#00000099] font-normal"
                                                } rounded-full text-[16px] leading-5 px-6 border-none h-10 sm:h-11`}
                                        >
                                            {s}
                                        </Button>
                                    );
                                })}
                            </div>
                        </div>

                        <Separator className="my-4 sm:my-6" />

                        {/* Quantity + Add to Cart */}
                        <div className="flex items-center gap-3 sm:gap-5">
                            <div className="flex items-center justify-between px-4 sm:px-5 h-12 sm:h-[52px] gap-2 min-w-[140px] sm:min-w-[170px] bg-[#F0F0F0] rounded-full">
                                <Button
                                    type="button"
                                    variant="outline"
                                    className="bg-transparent border-none shadow-none hover:bg-transparent p-0"
                                    onClick={() => setQty((q) => Math.max(1, q - 1))}
                                    disabled={disabledMinus}
                                >
                                    <Image src='/assets/icons/minus.svg' alt="minus" width={24} height={24} />
                                </Button>
                                <span className="text-[16px] leading-[22px] font-medium">{qty}</span>
                                <Button
                                    type="button"
                                    variant="outline"
                                    className="bg-transparent border-none shadow-none hover:bg-transparent p-0"
                                    onClick={() => setQty((q) => q + 1)}
                                >
                                    <Image src='/assets/icons/plus.svg' alt="plus" width={24} height={24} />
                                </Button>
                            </div>

                            <Button
                                type="button"
                                className="flex-1 rounded-full h-12 sm:h-[52px] px-8 md:px-8 text-[16px] font-medium"
                            >
                                Add to Cart
                            </Button>
                        </div>
                    </div>
                </div>
            </Container>
        </div>
    );
};

export default ProductDetails;
