"use client";

import React from "react";
import { Button } from "@/components/ui/button";
import { Menu, User } from "lucide-react";
import Link from "next/link";
import { usePathname } from "next/navigation";
import type { NavLink, NavbarProps } from "@/types/nav";
import {
    NavigationMenu,
    NavigationMenuList,
    NavigationMenuItem,
    NavigationMenuLink,
} from "@/components/ui/navigation-menu";
import { Sheet, SheetContent, SheetTrigger, SheetHeader, SheetTitle, SheetClose } from "@/components/ui/sheet";
import { Separator } from "@/components/ui/separator";
import Image from "next/image";
import Logo from "../Logo";
import CartDropdown from "../CartDropdown";
import { Container } from "../ui/container";
import { useAuth } from "@/context/AuthContext";

const defaultLinks: ReadonlyArray<NavLink> = [
    { href: "/", label: "الرئيسية" },
    { href: "/products", label: "منتجات" },
    { href: "/about", label: "من نحن" },
    // { href: "/cart", label: "عربة التسوق" },
    // { href: "/order", label: "الطلبات" },
] as const;

const Navbar: React.FC<NavbarProps> = ({ links = defaultLinks }) => {
    const pathname = usePathname();
    const { isAuthenticated, logout } = useAuth();

    const handleLogout = async () => {
        try {
            await logout();
        } catch (error) {
            console.error('Logout error:', error);
        }
    };

    return (
        <header className="w-full sticky top-0 z-40  bg-white border-b py-4 md:py-4 lg:py-4 px-4 lg:px-6">
            <Container className="flex items-center justify-between gap-4 lg:gap-6">
                <Logo/>
                
                <NavigationMenu className="hidden md:flex">
                    <NavigationMenuList className="flex items-center gap-0 lg:gap-[25px]">
                        {links.map((l) => {
                            const active = l.href === "/" ? pathname === "/" : pathname?.startsWith(l.href);
                            return (
                                <NavigationMenuItem key={l.href}>
                                    <NavigationMenuLink
                                        asChild
                                        className={""}
                                    >
                                        <Link
                                            href={l.href}
                                            aria-current={active ? "page" : undefined}
                                            className={(active ? "text-[#F5411C]" : "text-[#8E8E8E]") +
                                                " p-4 tracking-[1.3px] hover:bg-gray-100 uppercase text-[13px] leading-[17px] font-semibold"}
                                        >
                                            {l.label}
                                        </Link>
                                    </NavigationMenuLink>
                                </NavigationMenuItem>
                            );
                        })}
                    </NavigationMenuList>
                </NavigationMenu>

                <div className="flex items-center gap-4 lg:gap-6">
                    <CartDropdown className="hidden md:flex" />
                    {isAuthenticated ? (
                        <>
                            <Button 
                                variant="ghost" 
                                asChild
                                className="hidden md:inline-flex p-2 hover:bg-gray-100 rounded-full"
                            >
                                <Link href="/profile" aria-label="Profile">
                                    <User className="w-8 h-8 text-[#8E8E8E]" />
                                </Link>
                            </Button>
                            <Button 
                                variant="ghost" 
                                onClick={handleLogout}
                                className="hidden md:inline-flex text-[#8E8E8E] p-4 tracking-[1.3px] uppercase text-[13px] font-semibold"
                            >
                                Logout
                            </Button>
                        </>
                    ) : (
                        <>
                            <Button variant="ghost" asChild className="hidden md:inline-flex text-[#8E8E8E] p-4 tracking-[1.3px] uppercase text-[13px] font-semibold">
                                <Link href="/login">تسجيل الدخول</Link>
                            </Button>
                            <Button asChild className="hidden md:inline-flex rounded-full hover:bg-black/85 bg-black text-white py-4 px-6 lg:px-[37px] text-[16px] leading-[21px] font-medium">
                                <Link href="/register">
                                    إنشاء حساب
                                    <Image src="/assets/icons/arrow-down.svg" alt='arrow' width={18} height={18} />
                                </Link>
                            </Button>
                        </>
                    )}

                    <Sheet>
                        <SheetTrigger asChild>
                            <Button variant="ghost" className="py-2 md:hidden cursor-pointer">
                                <Menu className="size-9" />
                            </Button>
                        </SheetTrigger>
                        <SheetContent side="right" className="w-72 p-4">
                            <SheetHeader className="p-0 m-0">
                                <SheetTitle>
                                    <Logo/>
                                </SheetTitle>
                            </SheetHeader>
                            <div className="flex flex-col gap-1.5">
                                <div className="flex justify-center py-2">
                                    <CartDropdown />
                                </div>
                                <Separator className="my-1" />
                                {links.map((l) => {
                                    const active = l.href === "/" ? pathname === "/" : pathname?.startsWith(l.href);
                                    return (
                                        <SheetClose asChild key={l.href}>
                                            <Button
                                                variant="ghost"
                                                asChild
                                                className={(active ? "text-[#F5411C]" : "text-[#8E8E8E]") +
                                                    " w-full justify-start p-4 tracking-[1.3px] hover:bg-gray-100 uppercase text-[13px] leading-[17px] font-semibold rounded-md"}
                                                aria-current={active ? "page" : undefined}
                                            >
                                                <Link href={l.href}>{l.label}</Link>
                                            </Button>
                                        </SheetClose>
                                    );
                                })}
                                <Separator className="my-2" />
                                {isAuthenticated ? (
                                    <>
                                        <SheetClose asChild>
                                            <Button 
                                                variant="ghost" 
                                                asChild
                                                className="flex justify-start text-[#8E8E8E] p-4 tracking-[1.3px] uppercase text-[13px] font-semibold"
                                            >
                                                <Link href="/profile" className="flex items-center gap-2">
                                                    <User className="w-5 h-5" />
                                                    الملف الشخصي
                                                </Link>
                                            </Button>
                                        </SheetClose>
                                        <SheetClose asChild>
                                            <Button 
                                                variant="ghost" 
                                                onClick={handleLogout}
                                                className="flex text-[#8E8E8E] p-4 tracking-[1.3px] uppercase text-[13px] font-semibold"
                                            >
                                                Logout
                                            </Button>
                                        </SheetClose>
                                    </>
                                ) : (
                                    <>
                                        <SheetClose asChild>
                                            <Button variant="ghost" asChild className="flex text-[#8E8E8E] p-4 tracking-[1.3px] uppercase text-[13px] font-semibold">
                                                <Link href="/login">Login</Link>
                                            </Button>
                                        </SheetClose>
                                        <SheetClose asChild>
                                            <Button asChild className="flex rounded-full hover:bg-black/85 bg-black text-white py-4 px-6 lg:px-[37px] text-[16px] leading-[21px] font-medium">
                                                <Link href="/register">
                                                    Sign Up Now
                                                    <Image src="/assets/icons/arrow-down.svg" alt='arrow' width={18} height={18} />
                                                </Link>
                                            </Button>
                                        </SheetClose>
                                    </>
                                )}
                            </div>
                        </SheetContent>
                    </Sheet>
                </div>
            </Container>
        </header>
    );
}
    ;

export default Navbar;