'use client';

import React, { Suspense } from 'react';

/**
 * Loading skeleton for product cards
 */
function ProductCardSkeleton() {
  return (
    <div className="bg-white rounded-lg shadow-md overflow-hidden animate-pulse">
      <div className="aspect-square bg-gray-200" />
      <div className="p-4">
        <div className="h-4 bg-gray-200 rounded mb-2" />
        <div className="h-4 bg-gray-200 rounded w-3/4 mb-3" />
        <div className="h-6 bg-gray-200 rounded w-1/2" />
      </div>
    </div>
  );
}

/**
 * Loading skeleton for category cards
 */
function CategorySkeleton() {
  return (
    <div className="bg-white rounded-lg shadow-md p-6 animate-pulse">
      <div className="h-8 bg-gray-200 rounded mb-2" />
      <div className="h-4 bg-gray-200 rounded w-2/3" />
    </div>
  );
}

/**
 * Grid skeleton for multiple items
 */
function GridSkeleton({ 
  count = 8, 
  type = 'product' 
}: { 
  count?: number; 
  type?: 'product' | 'category';
}) {
  const SkeletonComponent = type === 'product' ? ProductCardSkeleton : CategorySkeleton;
  
  return (
    <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-6">
      {Array.from({ length: count }).map((_, index) => (
        <SkeletonComponent key={index} />
      ))}
    </div>
  );
}

/**
 * Section loading wrapper with title
 */
function SectionSkeleton({ 
  title, 
  count = 4,
  type = 'product'
}: {
  title: string;
  count?: number;
  type?: 'product' | 'category';
}) {
  return (
    <section className="mb-12">
      <div className="flex items-center justify-between mb-8">
        <h2 className="text-3xl font-bold text-gray-900">{title}</h2>
        <div className="h-6 w-24 bg-gray-200 rounded animate-pulse" />
      </div>
      <GridSkeleton count={count} type={type} />
    </section>
  );
}

/**
 * Hero section skeleton
 */
function HeroSkeleton() {
  return (
    <section className="relative h-96 bg-gray-200 rounded-lg mb-12 animate-pulse">
      <div className="absolute inset-0 flex items-center justify-center">
        <div className="text-center">
          <div className="h-8 bg-gray-300 rounded w-64 mb-4 mx-auto" />
          <div className="h-4 bg-gray-300 rounded w-48 mb-6 mx-auto" />
          <div className="h-10 bg-gray-300 rounded w-32 mx-auto" />
        </div>
      </div>
    </section>
  );
}

/**
 * Complete home page loading skeleton
 */
export function HomePageSkeleton() {
  return (
    <div className="min-h-screen bg-gray-50 py-8 px-2">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        {/* Hero Section */}
        <HeroSkeleton />
        
        {/* Categories Section */}
        <SectionSkeleton 
          title="Shop by Category" 
          count={6} 
          type="category" 
        />
        
        {/* Featured Products */}
        <SectionSkeleton 
          title="Featured Products" 
          count={8} 
          type="product" 
        />
        
        {/* New Arrivals */}
        <SectionSkeleton 
          title="وصل جديدنا"
          count={4} 
          type="product" 
        />
      </div>
    </div>
  );
}

/**
 * Optimized suspense wrapper
 */
export function OptimizedSuspense({ 
  children, 
  fallback,
  delay = 200 
}: {
  children: React.ReactNode;
  fallback?: React.ReactNode;
  delay?: number;
}) {
  const [showFallback, setShowFallback] = React.useState(false);

  React.useEffect(() => {
    const timer = setTimeout(() => {
      setShowFallback(true);
    }, delay);

    return () => clearTimeout(timer);
  }, [delay]);

  return (
    <Suspense fallback={showFallback ? fallback : null}>
      {children}
    </Suspense>
  );
}

export {
  ProductCardSkeleton,
  CategorySkeleton,
  GridSkeleton,
  SectionSkeleton,
  HeroSkeleton,
};