import React from 'react';
import { Button } from '@/components/ui/button';
import { ChevronLeft, ChevronRight } from 'lucide-react';

interface PaginationProps {
  currentPage: number;
  lastPage: number;
  total: number;
  from: number;
  to: number;
  onPageChange: (page: number) => void;
  loading?: boolean;
}

export function Pagination({
  currentPage,
  lastPage,
  total,
  from,
  to,
  onPageChange,
  loading = false
}: PaginationProps) {
  const generatePageNumbers = () => {
    const pages: (number | string)[] = [];
    const delta = 2; // Number of pages to show around current page

    // Always show first page
    pages.push(1);

    // Add ellipsis if there's a gap
    if (currentPage - delta > 2) {
      pages.push('...');
    }

    // Add pages around current page
    const start = Math.max(2, currentPage - delta);
    const end = Math.min(lastPage - 1, currentPage + delta);

    for (let i = start; i <= end; i++) {
      pages.push(i);
    }

    // Add ellipsis if there's a gap
    if (currentPage + delta < lastPage - 1) {
      pages.push('...');
    }

    // Always show last page (if more than 1 page)
    if (lastPage > 1) {
      pages.push(lastPage);
    }

    return pages;
  };

  const handlePageClick = (page: number) => {
    if (!loading && page !== currentPage && page >= 1 && page <= lastPage) {
      onPageChange(page);
    }
  };

  const handlePrevious = () => {
    if (currentPage > 1) {
      handlePageClick(currentPage - 1);
    }
  };

  const handleNext = () => {
    if (currentPage < lastPage) {
      handlePageClick(currentPage + 1);
    }
  };

  if (lastPage <= 1) return null;

  return (
    <div className="flex flex-col sm:flex-row items-center justify-between gap-4 mt-8">
      {/* Results Info */}
      <div className="text-sm text-gray-600">
        عرض <span className="font-medium">{from}</span> إلى{' '}
        <span className="font-medium">{to}</span> من{' '}
        <span className="font-medium">{total}</span> نتيجة
      </div>

      {/* Pagination Controls */}
      <div className="flex items-center gap-1">
        {/* Previous Button */}
        <Button
          variant="outline"
          size="sm"
          onClick={handlePrevious}
          disabled={currentPage === 1 || loading}
          className="flex items-center gap-1 px-3 py-2"
        >
          <ChevronLeft className="w-4 h-4" />
          السابق
        </Button>

        {/* Page Numbers */}
        <div className="flex items-center gap-1 mx-2">
          {generatePageNumbers().map((page, index) => (
            <React.Fragment key={index}>
              {page === '...' ? (
                <span className="px-3 py-2 text-gray-400">...</span>
              ) : (
                <Button
                  variant={currentPage === page ? "default" : "outline"}
                  size="sm"
                  onClick={() => handlePageClick(page as number)}
                  disabled={loading}
                  className={`min-w-[40px] h-10 ${
                    currentPage === page
                      ? 'bg-black text-white hover:bg-gray-800'
                      : 'hover:bg-gray-50'
                  }`}
                >
                  {page}
                </Button>
              )}
            </React.Fragment>
          ))}
        </div>

        {/* Next Button */}
        <Button
          variant="outline"
          size="sm"
          onClick={handleNext}
          disabled={currentPage === lastPage || loading}
          className="flex items-center gap-1 px-3 py-2"
        >
          التالي
          <ChevronRight className="w-4 h-4" />
        </Button>
      </div>
    </div>
  );
}

// Simple pagination for mobile
export function SimplePagination({
  currentPage,
  lastPage,
  onPageChange,
  loading = false
}: Pick<PaginationProps, 'currentPage' | 'lastPage' | 'onPageChange' | 'loading'>) {
  if (lastPage <= 1) return null;

  return (
    <div className="flex items-center justify-between gap-4 mt-6 sm:hidden">
      <Button
        variant="outline"
        onClick={() => onPageChange(currentPage - 1)}
        disabled={currentPage === 1 || loading}
        className="flex items-center gap-2"
      >
        <ChevronLeft className="w-4 h-4" />
        السابق
      </Button>

      <span className="text-sm text-gray-600">
        صفحة {currentPage} من {lastPage}
      </span>

      <Button
        variant="outline"
        onClick={() => onPageChange(currentPage + 1)}
        disabled={currentPage === lastPage || loading}
        className="flex items-center gap-2"
      >
        التالي
        <ChevronRight className="w-4 h-4" />
      </Button>
    </div>
  );
}